<?php
/**
 * This file is part of the Carbon package.
 *
 * (c) Brian Nesbitt <brian@nesbot.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace MailPoetVendor\Carbon;
if (!defined('ABSPATH')) exit;
use MailPoetVendor\Carbon\Exceptions\BadFluentConstructorException;
use MailPoetVendor\Carbon\Exceptions\BadFluentSetterException;
use MailPoetVendor\Carbon\Exceptions\InvalidCastException;
use MailPoetVendor\Carbon\Exceptions\InvalidIntervalException;
use MailPoetVendor\Carbon\Exceptions\ParseErrorException;
use MailPoetVendor\Carbon\Exceptions\UnitNotConfiguredException;
use MailPoetVendor\Carbon\Exceptions\UnknownGetterException;
use MailPoetVendor\Carbon\Exceptions\UnknownSetterException;
use MailPoetVendor\Carbon\Exceptions\UnknownUnitException;
use MailPoetVendor\Carbon\Traits\IntervalRounding;
use MailPoetVendor\Carbon\Traits\IntervalStep;
use MailPoetVendor\Carbon\Traits\Mixin;
use MailPoetVendor\Carbon\Traits\Options;
use MailPoetVendor\Carbon\Traits\ToStringFormat;
use Closure;
use DateInterval;
use DateTimeInterface;
use DateTimeZone;
use Exception;
use ReflectionException;
use ReturnTypeWillChange;
use Throwable;
/**
 * A simple API extension for DateInterval.
 * The implementation provides helpers to handle weeks but only days are saved.
 * Weeks are calculated based on the total days of the current instance.
 *
 * @property int $years Total years of the current interval.
 * @property int $months Total months of the current interval.
 * @property int $weeks Total weeks of the current interval calculated from the days.
 * @property int $dayz Total days of the current interval (weeks * 7 + days).
 * @property int $hours Total hours of the current interval.
 * @property int $minutes Total minutes of the current interval.
 * @property int $seconds Total seconds of the current interval.
 * @property int $microseconds Total microseconds of the current interval.
 * @property int $milliseconds Total microseconds of the current interval.
 * @property int $microExcludeMilli Remaining microseconds without the milliseconds.
 * @property int $dayzExcludeWeeks Total days remaining in the final week of the current instance (days % 7).
 * @property int $daysExcludeWeeks alias of dayzExcludeWeeks
 * @property-read float $totalYears Number of years equivalent to the interval.
 * @property-read float $totalMonths Number of months equivalent to the interval.
 * @property-read float $totalWeeks Number of weeks equivalent to the interval.
 * @property-read float $totalDays Number of days equivalent to the interval.
 * @property-read float $totalDayz Alias for totalDays.
 * @property-read float $totalHours Number of hours equivalent to the interval.
 * @property-read float $totalMinutes Number of minutes equivalent to the interval.
 * @property-read float $totalSeconds Number of seconds equivalent to the interval.
 * @property-read float $totalMilliseconds Number of milliseconds equivalent to the interval.
 * @property-read float $totalMicroseconds Number of microseconds equivalent to the interval.
 * @property-read string $locale locale of the current instance
 *
 * @method static CarbonInterval years($years = 1) Create instance specifying a number of years or modify the number of years if called on an instance.
 * @method static CarbonInterval year($years = 1) Alias for years()
 * @method static CarbonInterval months($months = 1) Create instance specifying a number of months or modify the number of months if called on an instance.
 * @method static CarbonInterval month($months = 1) Alias for months()
 * @method static CarbonInterval weeks($weeks = 1) Create instance specifying a number of weeks or modify the number of weeks if called on an instance.
 * @method static CarbonInterval week($weeks = 1) Alias for weeks()
 * @method static CarbonInterval days($days = 1) Create instance specifying a number of days or modify the number of days if called on an instance.
 * @method static CarbonInterval dayz($days = 1) Alias for days()
 * @method static CarbonInterval daysExcludeWeeks($days = 1) Create instance specifying a number of days or modify the number of days (keeping the current number of weeks) if called on an instance.
 * @method static CarbonInterval dayzExcludeWeeks($days = 1) Alias for daysExcludeWeeks()
 * @method static CarbonInterval day($days = 1) Alias for days()
 * @method static CarbonInterval hours($hours = 1) Create instance specifying a number of hours or modify the number of hours if called on an instance.
 * @method static CarbonInterval hour($hours = 1) Alias for hours()
 * @method static CarbonInterval minutes($minutes = 1) Create instance specifying a number of minutes or modify the number of minutes if called on an instance.
 * @method static CarbonInterval minute($minutes = 1) Alias for minutes()
 * @method static CarbonInterval seconds($seconds = 1) Create instance specifying a number of seconds or modify the number of seconds if called on an instance.
 * @method static CarbonInterval second($seconds = 1) Alias for seconds()
 * @method static CarbonInterval milliseconds($milliseconds = 1) Create instance specifying a number of milliseconds or modify the number of milliseconds if called on an instance.
 * @method static CarbonInterval millisecond($milliseconds = 1) Alias for milliseconds()
 * @method static CarbonInterval microseconds($microseconds = 1) Create instance specifying a number of microseconds or modify the number of microseconds if called on an instance.
 * @method static CarbonInterval microsecond($microseconds = 1) Alias for microseconds()
 * @method $this addYears(int $years) Add given number of years to the current interval
 * @method $this subYears(int $years) Subtract given number of years to the current interval
 * @method $this addMonths(int $months) Add given number of months to the current interval
 * @method $this subMonths(int $months) Subtract given number of months to the current interval
 * @method $this addWeeks(int|float $weeks) Add given number of weeks to the current interval
 * @method $this subWeeks(int|float $weeks) Subtract given number of weeks to the current interval
 * @method $this addDays(int|float $days) Add given number of days to the current interval
 * @method $this subDays(int|float $days) Subtract given number of days to the current interval
 * @method $this addHours(int|float $hours) Add given number of hours to the current interval
 * @method $this subHours(int|float $hours) Subtract given number of hours to the current interval
 * @method $this addMinutes(int|float $minutes) Add given number of minutes to the current interval
 * @method $this subMinutes(int|float $minutes) Subtract given number of minutes to the current interval
 * @method $this addSeconds(int|float $seconds) Add given number of seconds to the current interval
 * @method $this subSeconds(int|float $seconds) Subtract given number of seconds to the current interval
 * @method $this addMilliseconds(int|float $milliseconds) Add given number of milliseconds to the current interval
 * @method $this subMilliseconds(int|float $milliseconds) Subtract given number of milliseconds to the current interval
 * @method $this addMicroseconds(int|float $microseconds) Add given number of microseconds to the current interval
 * @method $this subMicroseconds(int|float $microseconds) Subtract given number of microseconds to the current interval
 * @method $this roundYear(int|float $precision = 1, string $function = "round") Round the current instance year with given precision using the given function.
 * @method $this roundYears(int|float $precision = 1, string $function = "round") Round the current instance year with given precision using the given function.
 * @method $this floorYear(int|float $precision = 1) Truncate the current instance year with given precision.
 * @method $this floorYears(int|float $precision = 1) Truncate the current instance year with given precision.
 * @method $this ceilYear(int|float $precision = 1) Ceil the current instance year with given precision.
 * @method $this ceilYears(int|float $precision = 1) Ceil the current instance year with given precision.
 * @method $this roundMonth(int|float $precision = 1, string $function = "round") Round the current instance month with given precision using the given function.
 * @method $this roundMonths(int|float $precision = 1, string $function = "round") Round the current instance month with given precision using the given function.
 * @method $this floorMonth(int|float $precision = 1) Truncate the current instance month with given precision.
 * @method $this floorMonths(int|float $precision = 1) Truncate the current instance month with given precision.
 * @method $this ceilMonth(int|float $precision = 1) Ceil the current instance month with given precision.
 * @method $this ceilMonths(int|float $precision = 1) Ceil the current instance month with given precision.
 * @method $this roundWeek(int|float $precision = 1, string $function = "round") Round the current instance day with given precision using the given function.
 * @method $this roundWeeks(int|float $precision = 1, string $function = "round") Round the current instance day with given precision using the given function.
 * @method $this floorWeek(int|float $precision = 1) Truncate the current instance day with given precision.
 * @method $this floorWeeks(int|float $precision = 1) Truncate the current instance day with given precision.
 * @method $this ceilWeek(int|float $precision = 1) Ceil the current instance day with given precision.
 * @method $this ceilWeeks(int|float $precision = 1) Ceil the current instance day with given precision.
 * @method $this roundDay(int|float $precision = 1, string $function = "round") Round the current instance day with given precision using the given function.
 * @method $this roundDays(int|float $precision = 1, string $function = "round") Round the current instance day with given precision using the given function.
 * @method $this floorDay(int|float $precision = 1) Truncate the current instance day with given precision.
 * @method $this floorDays(int|float $precision = 1) Truncate the current instance day with given precision.
 * @method $this ceilDay(int|float $precision = 1) Ceil the current instance day with given precision.
 * @method $this ceilDays(int|float $precision = 1) Ceil the current instance day with given precision.
 * @method $this roundHour(int|float $precision = 1, string $function = "round") Round the current instance hour with given precision using the given function.
 * @method $this roundHours(int|float $precision = 1, string $function = "round") Round the current instance hour with given precision using the given function.
 * @method $this floorHour(int|float $precision = 1) Truncate the current instance hour with given precision.
 * @method $this floorHours(int|float $precision = 1) Truncate the current instance hour with given precision.
 * @method $this ceilHour(int|float $precision = 1) Ceil the current instance hour with given precision.
 * @method $this ceilHours(int|float $precision = 1) Ceil the current instance hour with given precision.
 * @method $this roundMinute(int|float $precision = 1, string $function = "round") Round the current instance minute with given precision using the given function.
 * @method $this roundMinutes(int|float $precision = 1, string $function = "round") Round the current instance minute with given precision using the given function.
 * @method $this floorMinute(int|float $precision = 1) Truncate the current instance minute with given precision.
 * @method $this floorMinutes(int|float $precision = 1) Truncate the current instance minute with given precision.
 * @method $this ceilMinute(int|float $precision = 1) Ceil the current instance minute with given precision.
 * @method $this ceilMinutes(int|float $precision = 1) Ceil the current instance minute with given precision.
 * @method $this roundSecond(int|float $precision = 1, string $function = "round") Round the current instance second with given precision using the given function.
 * @method $this roundSeconds(int|float $precision = 1, string $function = "round") Round the current instance second with given precision using the given function.
 * @method $this floorSecond(int|float $precision = 1) Truncate the current instance second with given precision.
 * @method $this floorSeconds(int|float $precision = 1) Truncate the current instance second with given precision.
 * @method $this ceilSecond(int|float $precision = 1) Ceil the current instance second with given precision.
 * @method $this ceilSeconds(int|float $precision = 1) Ceil the current instance second with given precision.
 * @method $this roundMillennium(int|float $precision = 1, string $function = "round") Round the current instance millennium with given precision using the given function.
 * @method $this roundMillennia(int|float $precision = 1, string $function = "round") Round the current instance millennium with given precision using the given function.
 * @method $this floorMillennium(int|float $precision = 1) Truncate the current instance millennium with given precision.
 * @method $this floorMillennia(int|float $precision = 1) Truncate the current instance millennium with given precision.
 * @method $this ceilMillennium(int|float $precision = 1) Ceil the current instance millennium with given precision.
 * @method $this ceilMillennia(int|float $precision = 1) Ceil the current instance millennium with given precision.
 * @method $this roundCentury(int|float $precision = 1, string $function = "round") Round the current instance century with given precision using the given function.
 * @method $this roundCenturies(int|float $precision = 1, string $function = "round") Round the current instance century with given precision using the given function.
 * @method $this floorCentury(int|float $precision = 1) Truncate the current instance century with given precision.
 * @method $this floorCenturies(int|float $precision = 1) Truncate the current instance century with given precision.
 * @method $this ceilCentury(int|float $precision = 1) Ceil the current instance century with given precision.
 * @method $this ceilCenturies(int|float $precision = 1) Ceil the current instance century with given precision.
 * @method $this roundDecade(int|float $precision = 1, string $function = "round") Round the current instance decade with given precision using the given function.
 * @method $this roundDecades(int|float $precision = 1, string $function = "round") Round the current instance decade with given precision using the given function.
 * @method $this floorDecade(int|float $precision = 1) Truncate the current instance decade with given precision.
 * @method $this floorDecades(int|float $precision = 1) Truncate the current instance decade with given precision.
 * @method $this ceilDecade(int|float $precision = 1) Ceil the current instance decade with given precision.
 * @method $this ceilDecades(int|float $precision = 1) Ceil the current instance decade with given precision.
 * @method $this roundQuarter(int|float $precision = 1, string $function = "round") Round the current instance quarter with given precision using the given function.
 * @method $this roundQuarters(int|float $precision = 1, string $function = "round") Round the current instance quarter with given precision using the given function.
 * @method $this floorQuarter(int|float $precision = 1) Truncate the current instance quarter with given precision.
 * @method $this floorQuarters(int|float $precision = 1) Truncate the current instance quarter with given precision.
 * @method $this ceilQuarter(int|float $precision = 1) Ceil the current instance quarter with given precision.
 * @method $this ceilQuarters(int|float $precision = 1) Ceil the current instance quarter with given precision.
 * @method $this roundMillisecond(int|float $precision = 1, string $function = "round") Round the current instance millisecond with given precision using the given function.
 * @method $this roundMilliseconds(int|float $precision = 1, string $function = "round") Round the current instance millisecond with given precision using the given function.
 * @method $this floorMillisecond(int|float $precision = 1) Truncate the current instance millisecond with given precision.
 * @method $this floorMilliseconds(int|float $precision = 1) Truncate the current instance millisecond with given precision.
 * @method $this ceilMillisecond(int|float $precision = 1) Ceil the current instance millisecond with given precision.
 * @method $this ceilMilliseconds(int|float $precision = 1) Ceil the current instance millisecond with given precision.
 * @method $this roundMicrosecond(int|float $precision = 1, string $function = "round") Round the current instance microsecond with given precision using the given function.
 * @method $this roundMicroseconds(int|float $precision = 1, string $function = "round") Round the current instance microsecond with given precision using the given function.
 * @method $this floorMicrosecond(int|float $precision = 1) Truncate the current instance microsecond with given precision.
 * @method $this floorMicroseconds(int|float $precision = 1) Truncate the current instance microsecond with given precision.
 * @method $this ceilMicrosecond(int|float $precision = 1) Ceil the current instance microsecond with given precision.
 * @method $this ceilMicroseconds(int|float $precision = 1) Ceil the current instance microsecond with given precision.
 */
class CarbonInterval extends DateInterval implements CarbonConverterInterface
{
 use IntervalRounding;
 use IntervalStep;
 use Mixin {
 Mixin::mixin as baseMixin;
 }
 use Options;
 use ToStringFormat;
 /**
 * Interval spec period designators
 */
 public const PERIOD_PREFIX = 'P';
 public const PERIOD_YEARS = 'Y';
 public const PERIOD_MONTHS = 'M';
 public const PERIOD_DAYS = 'D';
 public const PERIOD_TIME_PREFIX = 'T';
 public const PERIOD_HOURS = 'H';
 public const PERIOD_MINUTES = 'M';
 public const PERIOD_SECONDS = 'S';
 /**
 * A translator to ... er ... translate stuff
 *
 * @var \Symfony\Component\Translation\TranslatorInterface
 */
 protected static $translator;
 /**
 * @var array|null
 */
 protected static $cascadeFactors;
 /**
 * @var array
 */
 protected static $formats = ['y' => 'y', 'Y' => 'y', 'o' => 'y', 'm' => 'm', 'n' => 'm', 'W' => 'weeks', 'd' => 'd', 'j' => 'd', 'z' => 'd', 'h' => 'h', 'g' => 'h', 'H' => 'h', 'G' => 'h', 'i' => 'i', 's' => 's', 'u' => 'micro', 'v' => 'milli'];
 /**
 * @var array|null
 */
 private static $flipCascadeFactors;
 /**
 * The registered macros.
 *
 * @var array
 */
 protected static $macros = [];
 /**
 * Timezone handler for settings() method.
 *
 * @var mixed
 */
 protected $tzName;
 /**
 * Set the instance's timezone from a string or object.
 *
 * @param \DateTimeZone|string $tzName
 *
 * @return static
 */
 public function setTimezone($tzName)
 {
 $this->tzName = $tzName;
 return $this;
 }
 /**
 * @internal
 *
 * Set the instance's timezone from a string or object and add/subtract the offset difference.
 *
 * @param \DateTimeZone|string $tzName
 *
 * @return static
 */
 public function shiftTimezone($tzName)
 {
 $this->tzName = $tzName;
 return $this;
 }
 /**
 * Mapping of units and factors for cascading.
 *
 * Should only be modified by changing the factors or referenced constants.
 *
 * @return array
 */
 public static function getCascadeFactors()
 {
 return static::$cascadeFactors ?: ['milliseconds' => [Carbon::MICROSECONDS_PER_MILLISECOND, 'microseconds'], 'seconds' => [Carbon::MILLISECONDS_PER_SECOND, 'milliseconds'], 'minutes' => [Carbon::SECONDS_PER_MINUTE, 'seconds'], 'hours' => [Carbon::MINUTES_PER_HOUR, 'minutes'], 'dayz' => [Carbon::HOURS_PER_DAY, 'hours'], 'weeks' => [Carbon::DAYS_PER_WEEK, 'dayz'], 'months' => [Carbon::WEEKS_PER_MONTH, 'weeks'], 'years' => [Carbon::MONTHS_PER_YEAR, 'months']];
 }
 private static function standardizeUnit($unit)
 {
 $unit = \rtrim($unit, 'sz') . 's';
 return $unit === 'days' ? 'dayz' : $unit;
 }
 private static function getFlipCascadeFactors()
 {
 if (!self::$flipCascadeFactors) {
 self::$flipCascadeFactors = [];
 foreach (static::getCascadeFactors() as $to => [$factor, $from]) {
 self::$flipCascadeFactors[self::standardizeUnit($from)] = [self::standardizeUnit($to), $factor];
 }
 }
 return self::$flipCascadeFactors;
 }
 /**
 * Set default cascading factors for ->cascade() method.
 *
 * @param array $cascadeFactors
 */
 public static function setCascadeFactors(array $cascadeFactors)
 {
 self::$flipCascadeFactors = null;
 static::$cascadeFactors = $cascadeFactors;
 }
 ///////////////////////////////////////////////////////////////////
 //////////////////////////// CONSTRUCTORS /////////////////////////
 ///////////////////////////////////////////////////////////////////
 /**
 * Create a new CarbonInterval instance.
 *
 * @param Closure|DateInterval|string|int|null $years
 * @param int|null $months
 * @param int|null $weeks
 * @param int|null $days
 * @param int|null $hours
 * @param int|null $minutes
 * @param int|null $seconds
 * @param int|null $microseconds
 *
 * @throws Exception when the interval_spec (passed as $years) cannot be parsed as an interval.
 */
 public function __construct($years = 1, $months = null, $weeks = null, $days = null, $hours = null, $minutes = null, $seconds = null, $microseconds = null)
 {
 if ($years instanceof Closure) {
 $this->step = $years;
 $years = null;
 }
 if ($years instanceof DateInterval) {
 parent::__construct(static::getDateIntervalSpec($years));
 $this->f = $years->f;
 self::copyNegativeUnits($years, $this);
 return;
 }
 $spec = $years;
 if (!\is_string($spec) || (float) $years || \preg_match('/^[\\d.]/', $years)) {
 $spec = static::PERIOD_PREFIX;
 $spec .= $years > 0 ? $years . static::PERIOD_YEARS : '';
 $spec .= $months > 0 ? $months . static::PERIOD_MONTHS : '';
 $specDays = 0;
 $specDays += $weeks > 0 ? $weeks * static::getDaysPerWeek() : 0;
 $specDays += $days > 0 ? $days : 0;
 $spec .= $specDays > 0 ? $specDays . static::PERIOD_DAYS : '';
 if ($hours > 0 || $minutes > 0 || $seconds > 0) {
 $spec .= static::PERIOD_TIME_PREFIX;
 $spec .= $hours > 0 ? $hours . static::PERIOD_HOURS : '';
 $spec .= $minutes > 0 ? $minutes . static::PERIOD_MINUTES : '';
 $spec .= $seconds > 0 ? $seconds . static::PERIOD_SECONDS : '';
 }
 if ($spec === static::PERIOD_PREFIX) {
 // Allow the zero interval.
 $spec .= '0' . static::PERIOD_YEARS;
 }
 }
 parent::__construct($spec);
 if ($microseconds !== null) {
 $this->f = $microseconds / Carbon::MICROSECONDS_PER_SECOND;
 }
 }
 /**
 * Returns the factor for a given source-to-target couple.
 *
 * @param string $source
 * @param string $target
 *
 * @return int|float|null
 */
 public static function getFactor($source, $target)
 {
 $source = self::standardizeUnit($source);
 $target = self::standardizeUnit($target);
 $factors = self::getFlipCascadeFactors();
 if (isset($factors[$source])) {
 [$to, $factor] = $factors[$source];
 if ($to === $target) {
 return $factor;
 }
 return $factor * static::getFactor($to, $target);
 }
 return null;
 }
 /**
 * Returns the factor for a given source-to-target couple if set,
 * else try to find the appropriate constant as the factor, such as Carbon::DAYS_PER_WEEK.
 *
 * @param string $source
 * @param string $target
 *
 * @return int|float|null
 */
 public static function getFactorWithDefault($source, $target)
 {
 $factor = self::getFactor($source, $target);
 if ($factor) {
 return $factor;
 }
 static $defaults = ['month' => ['year' => Carbon::MONTHS_PER_YEAR], 'week' => ['month' => Carbon::WEEKS_PER_MONTH], 'day' => ['week' => Carbon::DAYS_PER_WEEK], 'hour' => ['day' => Carbon::HOURS_PER_DAY], 'minute' => ['hour' => Carbon::MINUTES_PER_HOUR], 'second' => ['minute' => Carbon::SECONDS_PER_MINUTE], 'millisecond' => ['second' => Carbon::MILLISECONDS_PER_SECOND], 'microsecond' => ['millisecond' => Carbon::MICROSECONDS_PER_MILLISECOND]];
 return $defaults[$source][$target] ?? null;
 }
 /**
 * Returns current config for days per week.
 *
 * @return int|float
 */
 public static function getDaysPerWeek()
 {
 return static::getFactor('dayz', 'weeks') ?: Carbon::DAYS_PER_WEEK;
 }
 /**
 * Returns current config for hours per day.
 *
 * @return int|float
 */
 public static function getHoursPerDay()
 {
 return static::getFactor('hours', 'dayz') ?: Carbon::HOURS_PER_DAY;
 }
 /**
 * Returns current config for minutes per hour.
 *
 * @return int|float
 */
 public static function getMinutesPerHour()
 {
 return static::getFactor('minutes', 'hours') ?: Carbon::MINUTES_PER_HOUR;
 }
 /**
 * Returns current config for seconds per minute.
 *
 * @return int|float
 */
 public static function getSecondsPerMinute()
 {
 return static::getFactor('seconds', 'minutes') ?: Carbon::SECONDS_PER_MINUTE;
 }
 /**
 * Returns current config for microseconds per second.
 *
 * @return int|float
 */
 public static function getMillisecondsPerSecond()
 {
 return static::getFactor('milliseconds', 'seconds') ?: Carbon::MILLISECONDS_PER_SECOND;
 }
 /**
 * Returns current config for microseconds per second.
 *
 * @return int|float
 */
 public static function getMicrosecondsPerMillisecond()
 {
 return static::getFactor('microseconds', 'milliseconds') ?: Carbon::MICROSECONDS_PER_MILLISECOND;
 }
 /**
 * Create a new CarbonInterval instance from specific values.
 * This is an alias for the constructor that allows better fluent
 * syntax as it allows you to do CarbonInterval::create(1)->fn() rather than
 * (new CarbonInterval(1))->fn().
 *
 * @param int $years
 * @param int $months
 * @param int $weeks
 * @param int $days
 * @param int $hours
 * @param int $minutes
 * @param int $seconds
 * @param int $microseconds
 *
 * @throws Exception when the interval_spec (passed as $years) cannot be parsed as an interval.
 *
 * @return static
 */
 public static function create($years = 1, $months = null, $weeks = null, $days = null, $hours = null, $minutes = null, $seconds = null, $microseconds = null)
 {
 return new static($years, $months, $weeks, $days, $hours, $minutes, $seconds, $microseconds);
 }
 /**
 * Parse a string into a new CarbonInterval object according to the specified format.
 *
 * @example
 * ```
 * echo Carboninterval::createFromFormat('H:i', '1:30');
 * ```
 *
 * @param string $format Format of the $interval input string
 * @param string|null $interval Input string to convert into an interval
 *
 * @throws \Carbon\Exceptions\ParseErrorException when the $interval cannot be parsed as an interval.
 *
 * @return static
 */
 public static function createFromFormat(string $format, ?string $interval)
 {
 $instance = new static(0);
 $length = \mb_strlen($format);
 if (\preg_match('/s([,.])([uv])$/', $format, $match)) {
 $interval = \explode($match[1], $interval);
 $index = \count($interval) - 1;
 $interval[$index] = \str_pad($interval[$index], $match[2] === 'v' ? 3 : 6, '0');
 $interval = \implode($match[1], $interval);
 }
 $interval = $interval ?? '';
 for ($index = 0; $index < $length; $index++) {
 $expected = \mb_substr($format, $index, 1);
 $nextCharacter = \mb_substr($interval, 0, 1);
 $unit = static::$formats[$expected] ?? null;
 if ($unit) {
 if (!\preg_match('/^-?\\d+/', $interval, $match)) {
 throw new ParseErrorException('number', $nextCharacter);
 }
 $interval = \mb_substr($interval, \mb_strlen($match[0]));
 $instance->{$unit} += (int) $match[0];
 continue;
 }
 if ($nextCharacter !== $expected) {
 throw new ParseErrorException("'{$expected}'", $nextCharacter, 'Allowed substitutes for interval formats are ' . \implode(', ', \array_keys(static::$formats)) . "\n" . 'See https://php.net/manual/en/function.date.php for their meaning');
 }
 $interval = \mb_substr($interval, 1);
 }
 if ($interval !== '') {
 throw new ParseErrorException('end of string', $interval);
 }
 return $instance;
 }
 /**
 * Get a copy of the instance.
 *
 * @return static
 */
 public function copy()
 {
 $date = new static(0);
 $date->copyProperties($this);
 $date->step = $this->step;
 return $date;
 }
 /**
 * Get a copy of the instance.
 *
 * @return static
 */
 public function clone()
 {
 return $this->copy();
 }
 /**
 * Provide static helpers to create instances. Allows CarbonInterval::years(3).
 *
 * Note: This is done using the magic method to allow static and instance methods to
 * have the same names.
 *
 * @param string $method magic method name called
 * @param array $parameters parameters list
 *
 * @return static|null
 */
 public static function __callStatic($method, $parameters)
 {
 try {
 $interval = new static(0);
 $localStrictModeEnabled = $interval->localStrictModeEnabled;
 $interval->localStrictModeEnabled = \true;
 $result = static::hasMacro($method) ? static::bindMacroContext(null, function () use(&$method, &$parameters, &$interval) {
 return $interval->callMacro($method, $parameters);
 }) : $interval->{$method}(...$parameters);
 $interval->localStrictModeEnabled = $localStrictModeEnabled;
 return $result;
 } catch (BadFluentSetterException $exception) {
 if (Carbon::isStrictModeEnabled()) {
 throw new BadFluentConstructorException($method, 0, $exception);
 }
 return null;
 }
 }
 /**
 * Return the current context from inside a macro callee or a new one if static.
 *
 * @return static
 */
 protected static function this()
 {
 return \end(static::$macroContextStack) ?: new static(0);
 }
 /**
 * Creates a CarbonInterval from string.
 *
 * Format:
 *
 * Suffix | Unit | Example | DateInterval expression
 * -------|---------|---------|------------------------
 * y | years | 1y | P1Y
 * mo | months | 3mo | P3M
 * w | weeks | 2w | P2W
 * d | days | 28d | P28D
 * h | hours | 4h | PT4H
 * m | minutes | 12m | PT12M
 * s | seconds | 59s | PT59S
 *
 * e. g. `1w 3d 4h 32m 23s` is converted to 10 days 4 hours 32 minutes and 23 seconds.
 *
 * Special cases:
 * - An empty string will return a zero interval
 * - Fractions are allowed for weeks, days, hours and minutes and will be converted
 * and rounded to the next smaller value (caution: 0.5w = 4d)
 *
 * @param string $intervalDefinition
 *
 * @return static
 */
 public static function fromString($intervalDefinition)
 {
 if (empty($intervalDefinition)) {
 return new static(0);
 }
 $years = 0;
 $months = 0;
 $weeks = 0;
 $days = 0;
 $hours = 0;
 $minutes = 0;
 $seconds = 0;
 $milliseconds = 0;
 $microseconds = 0;
 $pattern = '/(\\d+(?:\\.\\d+)?)\\h*([^\\d\\h]*)/i';
 \preg_match_all($pattern, $intervalDefinition, $parts, \PREG_SET_ORDER);
 while ([$part, $value, $unit] = \array_shift($parts)) {
 $intValue = (int) $value;
 $fraction = (float) $value - $intValue;
 // Fix calculation precision
 switch (\round($fraction, 6)) {
 case 1:
 $fraction = 0;
 $intValue++;
 break;
 case 0:
 $fraction = 0;
 break;
 }
 switch ($unit === 'µs' ? 'µs' : \strtolower($unit)) {
 case 'millennia':
 case 'millennium':
 $years += $intValue * CarbonInterface::YEARS_PER_MILLENNIUM;
 break;
 case 'century':
 case 'centuries':
 $years += $intValue * CarbonInterface::YEARS_PER_CENTURY;
 break;
 case 'decade':
 case 'decades':
 $years += $intValue * CarbonInterface::YEARS_PER_DECADE;
 break;
 case 'year':
 case 'years':
 case 'y':
 case 'yr':
 case 'yrs':
 $years += $intValue;
 break;
 case 'quarter':
 case 'quarters':
 $months += $intValue * CarbonInterface::MONTHS_PER_QUARTER;
 break;
 case 'month':
 case 'months':
 case 'mo':
 case 'mos':
 $months += $intValue;
 break;
 case 'week':
 case 'weeks':
 case 'w':
 $weeks += $intValue;
 if ($fraction) {
 $parts[] = [null, $fraction * static::getDaysPerWeek(), 'd'];
 }
 break;
 case 'day':
 case 'days':
 case 'd':
 $days += $intValue;
 if ($fraction) {
 $parts[] = [null, $fraction * static::getHoursPerDay(), 'h'];
 }
 break;
 case 'hour':
 case 'hours':
 case 'h':
 $hours += $intValue;
 if ($fraction) {
 $parts[] = [null, $fraction * static::getMinutesPerHour(), 'm'];
 }
 break;
 case 'minute':
 case 'minutes':
 case 'm':
 $minutes += $intValue;
 if ($fraction) {
 $parts[] = [null, $fraction * static::getSecondsPerMinute(), 's'];
 }
 break;
 case 'second':
 case 'seconds':
 case 's':
 $seconds += $intValue;
 if ($fraction) {
 $parts[] = [null, $fraction * static::getMillisecondsPerSecond(), 'ms'];
 }
 break;
 case 'millisecond':
 case 'milliseconds':
 case 'milli':
 case 'ms':
 $milliseconds += $intValue;
 if ($fraction) {
 $microseconds += \round($fraction * static::getMicrosecondsPerMillisecond());
 }
 break;
 case 'microsecond':
 case 'microseconds':
 case 'micro':
 case 'µs':
 $microseconds += $intValue;
 break;
 default:
 throw new InvalidIntervalException(\sprintf('Invalid part %s in definition %s', $part, $intervalDefinition));
 }
 }
 return new static($years, $months, $weeks, $days, $hours, $minutes, $seconds, $milliseconds * Carbon::MICROSECONDS_PER_MILLISECOND + $microseconds);
 }
 /**
 * Creates a CarbonInterval from string using a different locale.
 *
 * @param string $interval interval string in the given language (may also contain English).
 * @param string|null $locale if locale is null or not specified, current global locale will be used instead.
 *
 * @return static
 */
 public static function parseFromLocale($interval, $locale = null)
 {
 return static::fromString(Carbon::translateTimeString($interval, $locale ?: static::getLocale(), 'en'));
 }
 private static function castIntervalToClass(DateInterval $interval, string $className)
 {
 $mainClass = DateInterval::class;
 if (!\is_a($className, $mainClass, \true)) {
 throw new InvalidCastException("{$className} is not a sub-class of {$mainClass}.");
 }
 $microseconds = $interval->f;
 $instance = new $className(static::getDateIntervalSpec($interval));
 if ($microseconds) {
 $instance->f = $microseconds;
 }
 if ($interval instanceof self && \is_a($className, self::class, \true)) {
 self::copyStep($interval, $instance);
 }
 self::copyNegativeUnits($interval, $instance);
 return $instance;
 }
 private static function copyNegativeUnits(DateInterval $from, DateInterval $to) : void
 {
 $to->invert = $from->invert;
 foreach (['y', 'm', 'd', 'h', 'i', 's'] as $unit) {
 if ($from->{$unit} < 0) {
 $to->{$unit} *= -1;
 }
 }
 }
 private static function copyStep(self $from, self $to) : void
 {
 $to->setStep($from->getStep());
 }
 /**
 * Cast the current instance into the given class.
 *
 * @param string $className The $className::instance() method will be called to cast the current object.
 *
 * @return DateInterval
 */
 public function cast(string $className)
 {
 return self::castIntervalToClass($this, $className);
 }
 /**
 * Create a CarbonInterval instance from a DateInterval one. Can not instance
 * DateInterval objects created from DateTime::diff() as you can't externally
 * set the $days field.
 *
 * @param DateInterval $interval
 *
 * @return static
 */
 public static function instance(DateInterval $interval)
 {
 return self::castIntervalToClass($interval, static::class);
 }
 /**
 * Make a CarbonInterval instance from given variable if possible.
 *
 * Always return a new instance. Parse only strings and only these likely to be intervals (skip dates
 * and recurrences). Throw an exception for invalid format, but otherwise return null.
 *
 * @param mixed|int|DateInterval|string|Closure|null $interval interval or number of the given $unit
 * @param string|null $unit if specified, $interval must be an integer
 *
 * @return static|null
 */
 public static function make($interval, $unit = null)
 {
 if ($unit) {
 $interval = "{$interval} " . Carbon::pluralUnit($unit);
 }
 if ($interval instanceof DateInterval) {
 return static::instance($interval);
 }
 if ($interval instanceof Closure) {
 return new static($interval);
 }
 if (!\is_string($interval)) {
 return null;
 }
 return static::makeFromString($interval);
 }
 protected static function makeFromString(string $interval)
 {
 $interval = \preg_replace('/\\s+/', ' ', \trim($interval));
 if (\preg_match('/^P[T\\d]/', $interval)) {
 return new static($interval);
 }
 if (\preg_match('/^(?:\\h*\\d+(?:\\.\\d+)?\\h*[a-z]+)+$/i', $interval)) {
 return static::fromString($interval);
 }
 /** @var static $interval */
 $interval = static::createFromDateString($interval);
 return !$interval || $interval->isEmpty() ? null : $interval;
 }
 protected function resolveInterval($interval)
 {
 if (!$interval instanceof self) {
 return self::make($interval);
 }
 return $interval;
 }
 /**
 * Sets up a DateInterval from the relative parts of the string.
 *
 * @param string $time
 *
 * @return static
 *
 * @link https://php.net/manual/en/dateinterval.createfromdatestring.php
 */
 #[\ReturnTypeWillChange]
 public static function createFromDateString($time)
 {
 $interval = @parent::createFromDateString(\strtr($time, [',' => ' ', ' and ' => ' ']));
 if ($interval instanceof DateInterval) {
 $interval = static::instance($interval);
 }
 return $interval;
 }
 ///////////////////////////////////////////////////////////////////
 ///////////////////////// GETTERS AND SETTERS /////////////////////
 ///////////////////////////////////////////////////////////////////
 /**
 * Get a part of the CarbonInterval object.
 *
 * @param string $name
 *
 * @throws UnknownGetterException
 *
 * @return int|float|string
 */
 public function get($name)
 {
 if (\str_starts_with($name, 'total')) {
 return $this->total(\substr($name, 5));
 }
 switch ($name) {
 case 'years':
 return $this->y;
 case 'months':
 return $this->m;
 case 'dayz':
 return $this->d;
 case 'hours':
 return $this->h;
 case 'minutes':
 return $this->i;
 case 'seconds':
 return $this->s;
 case 'milli':
 case 'milliseconds':
 return (int) (\round($this->f * Carbon::MICROSECONDS_PER_SECOND) / Carbon::MICROSECONDS_PER_MILLISECOND);
 case 'micro':
 case 'microseconds':
 return (int) \round($this->f * Carbon::MICROSECONDS_PER_SECOND);
 case 'microExcludeMilli':
 return (int) \round($this->f * Carbon::MICROSECONDS_PER_SECOND) % Carbon::MICROSECONDS_PER_MILLISECOND;
 case 'weeks':
 return (int) ($this->d / (int) static::getDaysPerWeek());
 case 'daysExcludeWeeks':
 case 'dayzExcludeWeeks':
 return $this->d % (int) static::getDaysPerWeek();
 case 'locale':
 return $this->getTranslatorLocale();
 default:
 throw new UnknownGetterException($name);
 }
 }
 /**
 * Get a part of the CarbonInterval object.
 *
 * @param string $name
 *
 * @throws UnknownGetterException
 *
 * @return int|float|string
 */
 public function __get($name)
 {
 return $this->get($name);
 }
 /**
 * Set a part of the CarbonInterval object.
 *
 * @param string|array $name
 * @param int $value
 *
 * @throws UnknownSetterException
 *
 * @return $this
 */
 public function set($name, $value = null)
 {
 $properties = \is_array($name) ? $name : [$name => $value];
 foreach ($properties as $key => $value) {
 switch (Carbon::singularUnit(\rtrim($key, 'z'))) {
 case 'year':
 $this->y = $value;
 break;
 case 'month':
 $this->m = $value;
 break;
 case 'week':
 $this->d = $value * (int) static::getDaysPerWeek();
 break;
 case 'day':
 $this->d = $value;
 break;
 case 'daysexcludeweek':
 case 'dayzexcludeweek':
 $this->d = $this->weeks * (int) static::getDaysPerWeek() + $value;
 break;
 case 'hour':
 $this->h = $value;
 break;
 case 'minute':
 $this->i = $value;
 break;
 case 'second':
 $this->s = $value;
 break;
 case 'milli':
 case 'millisecond':
 $this->microseconds = $value * Carbon::MICROSECONDS_PER_MILLISECOND + $this->microseconds % Carbon::MICROSECONDS_PER_MILLISECOND;
 break;
 case 'micro':
 case 'microsecond':
 $this->f = $value / Carbon::MICROSECONDS_PER_SECOND;
 break;
 default:
 if ($this->localStrictModeEnabled ?? Carbon::isStrictModeEnabled()) {
 throw new UnknownSetterException($key);
 }
 $this->{$key} = $value;
 }
 }
 return $this;
 }
 /**
 * Set a part of the CarbonInterval object.
 *
 * @param string $name
 * @param int $value
 *
 * @throws UnknownSetterException
 */
 public function __set($name, $value)
 {
 $this->set($name, $value);
 }
 /**
 * Allow setting of weeks and days to be cumulative.
 *
 * @param int $weeks Number of weeks to set
 * @param int $days Number of days to set
 *
 * @return static
 */
 public function weeksAndDays($weeks, $days)
 {
 $this->dayz = $weeks * static::getDaysPerWeek() + $days;
 return $this;
 }
 /**
 * Returns true if the interval is empty for each unit.
 *
 * @return bool
 */
 public function isEmpty()
 {
 return $this->years === 0 && $this->months === 0 && $this->dayz === 0 && !$this->days && $this->hours === 0 && $this->minutes === 0 && $this->seconds === 0 && $this->microseconds === 0;
 }
 /**
 * Register a custom macro.
 *
 * @example
 * ```
 * CarbonInterval::macro('twice', function () {
 * return $this->times(2);
 * });
 * echo CarbonInterval::hours(2)->twice();
 * ```
 *
 * @param string $name
 * @param object|callable $macro
 *
 * @return void
 */
 public static function macro($name, $macro)
 {
 static::$macros[$name] = $macro;
 }
 /**
 * Register macros from a mixin object.
 *
 * @example
 * ```
 * CarbonInterval::mixin(new class {
 * public function daysToHours() {
 * return function () {
 * $this->hours += $this->days;
 * $this->days = 0;
 *
 * return $this;
 * };
 * }
 * public function hoursToDays() {
 * return function () {
 * $this->days += $this->hours;
 * $this->hours = 0;
 *
 * return $this;
 * };
 * }
 * });
 * echo CarbonInterval::hours(5)->hoursToDays() . "\n";
 * echo CarbonInterval::days(5)->daysToHours() . "\n";
 * ```
 *
 * @param object|string $mixin
 *
 * @throws ReflectionException
 *
 * @return void
 */
 public static function mixin($mixin)
 {
 static::baseMixin($mixin);
 }
 /**
 * Check if macro is registered.
 *
 * @param string $name
 *
 * @return bool
 */
 public static function hasMacro($name)
 {
 return isset(static::$macros[$name]);
 }
 /**
 * Call given macro.
 *
 * @param string $name
 * @param array $parameters
 *
 * @return mixed
 */
 protected function callMacro($name, $parameters)
 {
 $macro = static::$macros[$name]                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   